<?php

namespace App\Http\Controllers\Admin;

use App\Constants\Status;
use App\Http\Controllers\Controller;
use App\Models\Frontend;
use App\Rules\FileTypeValidate;
use Illuminate\Http\Request;
use App\Lib\RequiredConfig;

class GeneralSettingController extends Controller
{
    public function systemSetting()
    {
        $pageTitle = 'System Settings';
        $settings  = json_decode(file_get_contents(resource_path('views/admin/setting/settings.json')));
        return view('admin.setting.system', compact('pageTitle', 'settings'));
    }
    public function general()
    {
        $pageTitle       = 'General Setting';
        $timezones       = timezone_identifiers_list();
        $currentTimezone = array_search(config('app.timezone'), $timezones);
        return view('admin.setting.general', compact('pageTitle', 'timezones', 'currentTimezone'));
    }

    public function generalUpdate(Request $request)
    {
        $request->validate([
            'site_name'                       => 'required|string|max:40',
            'cur_text'                        => 'required|string|max:40',
            'cur_sym'                         => 'required|string|max:40',
            'base_color'                      => 'nullable|regex:/^[a-f0-9]{6}$/i',
            'secondary_color'                 => 'nullable|regex:/^[a-f0-9]{6}$/i',
            'timezone'                        => 'required|integer',
            'currency_format'                 => 'required|in:1,2,3',
            'paginate_number'                 => 'required|integer',
            'register_bonus'                  => 'required|numeric|gt:0',
            'demo_balance'                    => 'required|numeric|gt:0',
            'min_balance'                     => 'required|numeric|gt:0',
            'balance_transfer_fixed_charge'   => 'required|numeric|gte:0',
            'balance_transfer_percent_charge' => 'required|numeric|gte:0',
            'gameplay_number_for_bonus'       => 'required|numeric|gt:0',
            'gameplay_bonus'                  => 'required|numeric|gt:0',
        ]);

        $timezones = timezone_identifiers_list();
        $timezone = isset($timezones[$request->timezone]) ? $timezones[$request->timezone] : 'UTC';

        $general                                  = gs();
        $general->site_name                       = $request->site_name;
        $general->cur_text                        = $request->cur_text;
        $general->cur_sym                         = $request->cur_sym;
        $general->paginate_number                 = $request->paginate_number;
        $general->base_color                      = str_replace('#', '', $request->base_color);
        $general->secondary_color                 = str_replace('#', '', $request->secondary_color);
        $general->currency_format                 = $request->currency_format;
        $general->register_bonus                  = $request->register_bonus;
        $general->demo_balance                    = $request->demo_balance;
        $general->min_balance                     = $request->min_balance;
        $general->balance_transfer_fixed_charge   = $request->balance_transfer_fixed_charge;
        $general->balance_transfer_percent_charge = $request->balance_transfer_percent_charge;
        $general->gameplay_number_for_bonus       = $request->gameplay_number_for_bonus;
        $general->gameplay_bonus                  = $request->gameplay_bonus;
        $general->save();

        $timezoneFile = config_path('timezone.php');
        $content      = '<?php $timezone = "' . $timezone . '" ?>';
        file_put_contents($timezoneFile, $content);
        RequiredConfig::configured('general_setting');
        $notify[] = ['success', 'General setting updated successfully'];
        return back()->withNotify($notify);
    }

    public function systemConfiguration()
    {
        $pageTitle = 'System Configuration';
        return view('admin.setting.configuration', compact('pageTitle'));
    }

    public function systemConfigurationSubmit(Request $request)
    {
        $general                  = gs();
        $general->kv               = $request->kv ? Status::ENABLE : Status::DISABLE;
        $general->ev               = $request->ev ? Status::ENABLE : Status::DISABLE;
        $general->en               = $request->en ? Status::ENABLE : Status::DISABLE;
        $general->sv               = $request->sv ? Status::ENABLE : Status::DISABLE;
        $general->sn               = $request->sn ? Status::ENABLE : Status::DISABLE;
        $general->pn               = $request->pn ? Status::ENABLE : Status::DISABLE;
        $general->force_ssl        = $request->force_ssl ? Status::ENABLE : Status::DISABLE;
        $general->secure_password  = $request->secure_password ? Status::ENABLE : Status::DISABLE;
        $general->registration     = $request->registration ? Status::ENABLE : Status::DISABLE;
        $general->agree            = $request->agree ? Status::ENABLE : Status::DISABLE;
        $general->multi_language   = $request->multi_language ? Status::ENABLE : Status::DISABLE;
        $general->dc               = $request->dc ? Status::ENABLE : Status::DISABLE;
        $general->rb               = $request->rb ? Status::ENABLE : Status::DISABLE;
        $general->game_play        = $request->game_play ? Status::ENABLE : Status::DISABLE;
        $general->balance_transfer = $request->balance_transfer ? Status::ENABLE : Status::DISABLE;
        $general->save();
        $notify[] = ['success', 'System configuration updated successfully'];
        return back()->withNotify($notify);
    }

    public function logoIcon()
    {
        $pageTitle = 'Logo & Favicon';
        return view('admin.setting.logo_icon', compact('pageTitle'));
    }

    public function logoIconUpdate(Request $request)
    {
        $request->validate([
            'logo'    => ['image', new FileTypeValidate(['jpg', 'jpeg', 'png'])],
            'favicon' => ['image', new FileTypeValidate(['png'])],
            'pwa_thumb'   => ['image', new FileTypeValidate(['jpg', 'jpeg', 'png'])],
            'pwa_favicon' => ['image', new FileTypeValidate(['jpg', 'jpeg', 'png'])],
        ]);
        if ($request->hasFile('logo')) {
            $this->uploadImage($request->logo, 'logo.png');
        }
        if ($request->hasFile('favicon')) {
            $this->uploadImage($request->favicon, 'favicon.png');
        }
        if ($request->hasFile('pwa_thumb')) {
            $this->uploadImage($request->pwa_thumb, "pwa_thumb.png", getFileSize('pwa_thumb'));
        }
        if ($request->hasFile('pwa_favicon')) {
            $this->uploadImage($request->pwa_favicon, "pwa_favicon.png", getFileSize('pwa_favicon'));
        }

        RequiredConfig::configured('logo_favicon');

        $notify[] = ['success', 'Logo & favicon updated successfully'];
        return back()->withNotify($notify);
    }

    public function customCss()
    {
        $pageTitle   = 'Custom CSS';
        $file        = activeTemplate(true) . 'css/custom.css';
        if (file_exists($file)) {
            $fileContent = file_get_contents($file);
        } else {
            $fileContent = null;
        }
        return view('admin.setting.custom_css', compact('pageTitle', 'fileContent'));
    }

    public function sitemap()
    {
        $pageTitle   = 'Sitemap XML';
        $file        = 'sitemap.xml';
        if (file_exists($file)) {
            $fileContent = file_get_contents($file);
        } else {
            $fileContent = null;
        }
        return view('admin.setting.sitemap', compact('pageTitle', 'fileContent'));
    }

    public function sitemapSubmit(Request $request)
    {
        $file = 'sitemap.xml';
        if (!file_exists($file)) {
            fopen($file, "w");
        }
        file_put_contents($file, $request->sitemap);
        $notify[] = ['success', 'Sitemap updated successfully'];
        return back()->withNotify($notify);
    }

    public function robot()
    {
        $pageTitle   = 'Robots TXT';
        $file        = 'robots.xml';
        if (file_exists($file)) {
            $fileContent = file_get_contents($file);
        } else {
            $fileContent = null;
        }
        return view('admin.setting.robots', compact('pageTitle', 'fileContent'));
    }

    public function robotSubmit(Request $request)
    {
        $file = 'robots.xml';
        if (!file_exists($file)) {
            fopen($file, "w");
        }
        file_put_contents($file, $request->robots);
        $notify[] = ['success', 'Robots txt updated successfully'];
        return back()->withNotify($notify);
    }

    public function customCssSubmit(Request $request)
    {
        $file = activeTemplate(true) . 'css/custom.css';
        if (!file_exists($file)) {
            fopen($file, "w");
        }
        file_put_contents($file, $request->css);
        $notify[] = ['success', 'CSS updated successfully'];
        return back()->withNotify($notify);
    }

    public function maintenanceMode()
    {
        $pageTitle   = 'Maintenance Mode';
        $maintenance = Frontend::where('data_keys', 'maintenance.data')->firstOrFail();
        return view('admin.setting.maintenance', compact('pageTitle', 'maintenance'));
    }

    public function maintenanceModeSubmit(Request $request)
    {
        $request->validate([
            'description' => 'required',
            'image'       => ['nullable', new FileTypeValidate(['jpg', 'jpeg', 'png'])],
        ]);
        $general                   = gs();
        $general->maintenance_mode = $request->status ? Status::ENABLE : Status::DISABLE;
        $general->save();

        $maintenance = Frontend::where('data_keys', 'maintenance.data')->firstOrFail();
        $image = $maintenance?->data_values?->image;
        if ($request->hasFile('image')) {
            try {
                $old   = $image;
                $image = fileUploader($request->image, getFilePath('maintenance'), getFileSize('maintenance'), $old);
            } catch (\Exception $exp) {
                $notify[] = ['error', 'Couldn\'t upload your image'];
                return back()->withNotify($notify);
            }
        }

        $maintenance->data_values = [
            'description' => $request->description,
            'image'       => $image,
        ];
        $maintenance->save();

        $notify[] = ['success', 'Maintenance mode updated successfully'];
        return back()->withNotify($notify);
    }

    public function cookie()
    {
        $pageTitle = 'GDPR Cookie';
        $cookie    = Frontend::where('data_keys', 'cookie.data')->firstOrFail();
        return view('admin.setting.cookie', compact('pageTitle', 'cookie'));
    }

    public function cookieSubmit(Request $request)
    {
        $request->validate([
            'short_desc'  => 'required|string|max:255',
            'description' => 'required',
        ]);
        $cookie              = Frontend::where('data_keys', 'cookie.data')->firstOrFail();
        $cookie->data_values = [
            'short_desc'  => $request->short_desc,
            'description' => $request->description,
            'status'      => $request->status ? Status::ENABLE : Status::DISABLE,
        ];
        $cookie->save();
        $notify[] = ['success', 'Cookie policy updated successfully'];
        return back()->withNotify($notify);
    }

    public function socialiteCredentials()
    {
        $pageTitle = 'Social Login Credentials';
        return view('admin.setting.social_credential', compact('pageTitle'));
    }

    public function updateSocialiteCredentialStatus($key)
    {
        $general     = gs();
        $credentials = $general->socialite_credentials;
        try {
            $credentials->$key->status = $credentials->$key->status == Status::ENABLE ? Status::DISABLE : Status::ENABLE;
        } catch (\Throwable $th) {
            abort(404);
        }

        $general->socialite_credentials = $credentials;
        $general->save();

        $notify[] = ['success', 'Status changed successfully'];
        return back()->withNotify($notify);
    }

    public function updateSocialiteCredential(Request $request, $key)
    {
        $general     = gs();
        $credentials = $general->socialite_credentials;
        try {
            $credentials->$key->client_id = $request->client_id;
            $credentials->$key->client_secret = $request->client_secret;
        } catch (\Throwable $th) {
            abort(404);
        }
        $general->socialite_credentials = $credentials;
        $general->save();

        $notify[] = ['success', ucfirst($key) . ' credential updated successfully'];
        return back()->withNotify($notify);
    }

    private function uploadImage($file, $fileName, $resize = null)
    {
        try {
            $path = getFilePath('logoIcon');
            fileUploader($file, $path, $resize, filename: $fileName);
        } catch (\Exception $exp) {
            return back()->withErrors(["Couldn\'t upload the $fileName"]);
        }
    }
}
